/* 
 * Copyright (C) 2006-2021 Registro.br. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 * 1. Redistribution of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY REGISTRO.BR ``AS IS AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIE OF FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL REGISTRO.BR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */
/* $Id$ */
/** @file DefRegFunctions.H
 *  @brief EPP Defensive Registration related functions
 */

#ifndef __DEF_REG_FUNCTIONS_H__
#define __DEF_REG_FUNCTIONS_H__

#include "SheppCommandFunctions.H"
#include "SheppStrUtil.H"

#include "DefRegCheck.H"
#include "DefRegInfo.H"
#include "DefRegTransfer.H"
#include "DefRegCreate.H"
#include "DefRegDelete.H"
#include "DefRegRenew.H"
#include "DefRegUpdate.H"

/// print defensive registration command usage info
/**
   @param error_msg   error message to be printed
   @param specific    optional, choses specific sub-command
   @return 0 if ok, -1 otherwise
*/
int cmd_def_reg_help(string error_msg, string specific = "")
{
	if (error_msg != "") {
		printf("error: %s\n", error_msg.c_str());
	}

	printf("defensive registration command syntax help:\n");
	printf("\n");

	if (specific == "" || specific == "check") {
		printf("  check <name1:level> [name2:level ... nameN:level]\n");
		printf("\n");
	}

	if (specific == "" || specific == "info") {
		printf("  info <roid> [-auth authInfoPw[:authInfoRoid]]\n");
		printf("\n");
	}

	if (specific == "" || specific == "transfer") {
		printf("  transfer <query|request> <roid> \n"
		       "    [-period time:unit] [-auth authInfoPw[:authInfoRoid]]\n");
		printf("\n");
	}

	if (specific == "" || specific == "create") {
		printf("  create <name:level> <registrant> <adminContact>\n"
		       "    <-auth authInfoPw[:authInfoRoid]>\n"
		       "    [-tm trademarkId] [-tmc trademarkCountry]\n"
		       "    [-tmd trademarkDate] [-period time:unit]\n");
		printf("\n");
	}

	if (specific == "" || specific == "delete") {
		printf("  delete <roid>\n");
		printf("\n");
	}

	if (specific == "" || specific == "renew") {
		printf("  renew <roid> <-expdate curExpDate> [-period time:unit]\n");
		printf("\n");
	}

	if (specific == "" || specific == "update") {
		printf("  update <roid>\n"
		       "    [-add-status status:lang:\"description\" ...]\n"
		       "    [-rem-status status ...]\n"
		       "    [-r registrant] [-a adminContact]\n"
		       "    [-tm trademarkId] [-tmc trademarkCountry]\n"
		       "    [-tmd trademarkDate] [-auth authInfoPw[:authInfoRoid]]\n");
		printf("\n");
	}

	if (error_msg != "") {
		return -1;
	}

	return 0;
}

/// defensive registration check command function
/**
   @param args   vector with command-line arguments
   @return 0 if ok, -1 otherwise
*/
int cmd_def_reg_check(vector<string> &args)
{
	DefRegCheck act;
	DefRegCheckCmd *cmd = act.get_command();
	string cmd_name = "check";

	while (!args.empty()) {
		string nameStr;
		string levelStr;

		if (SheppStrUtil::split(args[0], nameStr, levelStr, ":") != 0) {
			return cmd_def_reg_help("invalid syntax near \"" + args[0] + "\"", cmd_name);
		}

		DefRegLevel::Value level = DefRegLevel::fromStr(levelStr);
		if (level == DefRegLevel::NONE) {
			return cmd_def_reg_help("invalid level \"" + levelStr + "\"");
		}

		DefRegName name;
		name.set_name(nameStr);
		name.set_level(level);

		cmd->insert_name(name);
		args.erase(args.begin());
	}

	if (cmd->get_name_list().empty()) {
		return cmd_def_reg_help("no names specified", cmd_name);
	}

	if (_debug) {
		set<DefRegName> names = cmd->get_name_list();
		set<DefRegName>::const_iterator it;
		printf("names to be checked:\n");
		for (it = names.begin(); it != names.end(); it++) {
			printf("  [%s:%s]\n",
			       it->get_name().c_str(),
			       DefRegLevel::toStr(it->get_level()).c_str());
		}
	}

	if (process_action(act) != 0) {
		return -1;
	}

	return 0;
}

/// defensive registration info command function
/**
   @param args   vector with command-line arguments
   @return 0 if ok, -1 otherwise
*/
int cmd_def_reg_info(vector<string> &args)
{
	DefRegInfo act;
	DefRegInfoCmd *cmd = act.get_command();
	string cmd_name = "info";

	if (args.empty()) {
		return cmd_def_reg_help("no roid specified", cmd_name);
	}

	cmd->set_roid(args[0]);
	args.erase(args.begin());

	while (!args.empty()) {
		if (args[0] == "-auth") {
			// set authInfo
			AuthInfo auth;
			if (SheppObjSet::authInfo(auth, args) != 0) {
				return cmd_def_reg_help("invalid auth", cmd_name);
			}
			cmd->set_authInfo(auth);
		} else {
			return cmd_def_reg_help("invalid syntax near \"" + args[0] + "\"",
			                        cmd_name);
		}
	}

	if (_debug) {
		printf("roid to get info: [%s]\n", cmd->get_roid().c_str());
		SheppPrint::authInfo(cmd->get_authInfo());
	} // _debug

	if (process_action(act) != 0) {
		return -1;
	}

	return 0;
}

/// defensive registration transfer command function
/**
   @param args   vector with command-line arguments
   @return 0 if ok, -1 otherwise
*/
int cmd_def_reg_transfer(vector<string> &args)
{
	DefRegTransfer act;
	DefRegTransferCmd *cmd = act.get_command();
	string cmd_name = "transfer";

	if (args.empty()) {
		return cmd_def_reg_help("no operation specified", cmd_name);
	}

	TransferOperation::Value op = TransferOperation::fromStr(args[0]);
	if (op == TransferOperation::NONE) {
		return cmd_def_reg_help("invalid operation", cmd_name);
	}

	cmd->set_operation(op);
	args.erase(args.begin());

	if (args.empty()) {
		return cmd_def_reg_help("no roid specified", cmd_name);
	}

	cmd->set_roid(args[0]);
	args.erase(args.begin());

	while (!args.empty()) {
		if (args[0] == "-period") {
			// period
			args.erase(args.begin());
			if (args.empty()) {
				return cmd_domain_help("period parameter missing", cmd_name);
			}

			string time;
			string unit;
			if (SheppStrUtil::split(args[0], time, unit, ":", false) != 0) {
				return cmd_domain_help("invalid period", cmd_name);
			}

			cmd->set_period(atoi(time.c_str()), unit);
			args.erase(args.begin());
		} else if (args[0] == "-auth") {
			// set authInfo
			AuthInfo auth;
			if (SheppObjSet::authInfo(auth, args) != 0) {
				return cmd_def_reg_help("invalid auth", cmd_name);
			}
			cmd->set_authInfo(auth);
		} else {
			return cmd_def_reg_help("invalid syntax near \"" + args[0] + "\"",
			                        cmd_name);
		}
	}

	if (_debug) {
		printf("defensive registration to transfer: [%s]\n", cmd->get_roid().c_str());
		SheppPrint::authInfo(cmd->get_authInfo());
	} // _debug

	if (process_action(act) != 0) {
		return -1;
	}

	return 0;
}

/// defensive registration create command function
/**
   @param args   vector with command-line arguments
   @return 0 if ok, -1 otherwise
*/
int cmd_def_reg_create(vector<string> &args)
{
	DefRegCreate act;
	DefRegCreateCmd *cmd = act.get_command();
	string cmd_name = "create";

	if (args.empty()) {
		return cmd_def_reg_help("no name specified", cmd_name);
	}

	string nameStr;
	string levelStr;

	if (SheppStrUtil::split(args[0], nameStr, levelStr, ":") != 0) {
		return cmd_def_reg_help("invalid syntax near \"" + args[0] + "\"", cmd_name);
	}

	DefRegLevel::Value level = DefRegLevel::fromStr(levelStr);
	if (level == DefRegLevel::NONE) {
		return cmd_def_reg_help("invalid level \"" + levelStr + "\"");
	}

	DefRegName name;
	name.set_name(nameStr);
	name.set_level(level);

	cmd->set_name(name);
	args.erase(args.begin());

	if (args.empty()) {
		return cmd_def_reg_help("no registrant specified", cmd_name);
	}

	cmd->set_registrant(args[0]);
	args.erase(args.begin());

	if (args.empty()) {
		return cmd_def_reg_help("no admin contact specified", cmd_name);
	}

	cmd->set_admin_contact(args[0]);
	args.erase(args.begin());

	while (!args.empty()) {
		if (args[0] == "-tm") {
			//trademark
			args.erase(args.begin());
			if (args.empty()) {
				return cmd_domain_help("trademark parameter missing", cmd_name);
			}

			cmd->set_trademark_id(args[0]);
			args.erase(args.begin());

		} else if (args[0] == "-tmc") {
			//trademark country
			args.erase(args.begin());
			if (args.empty()) {
				return cmd_domain_help("trademark country parameter missing", cmd_name);
			}

			cmd->set_trademark_country(args[0]);
			args.erase(args.begin());

		} else if (args[0] == "-tmd") {
			//trademark date
			args.erase(args.begin());
			if (args.empty()) {
				return cmd_domain_help("trademark date parameter missing", cmd_name);
			}

			cmd->set_trademark_date(args[0]);
			args.erase(args.begin());

		} else if (args[0] == "-period") {
			//registration period
			args.erase(args.begin());
			if (args.empty()) {
				return cmd_domain_help("period parameter missing", cmd_name);
			}

			string time;
			string unit;
			if (SheppStrUtil::split(args[0], time, unit, ":", false) != 0) {
				return cmd_domain_help("invalid period", cmd_name);
			}

			cmd->set_period(atoi(time.c_str()), unit);
			args.erase(args.begin());
		} else if (args[0] == "-auth") {
			// set authInfo
			AuthInfo auth;
			if (SheppObjSet::authInfo(auth, args) != 0) {
				return cmd_def_reg_help("invalid auth", cmd_name);
			}
			cmd->set_authInfo(auth);
		} else {
			return cmd_def_reg_help("invalid syntax near \"" + args[0] + "\"",
			                        cmd_name);
		}
	}

	if (cmd->get_authInfo().get_pw().empty()) {
		return cmd_def_reg_help("no auth info specified", cmd_name);
	}

	if (_debug) {
		DefRegName name = cmd->get_name();
		printf("creating defensive registration: [%s:%s]\n", 
		       name.get_name().c_str(), 
		       DefRegLevel::toStr(name.get_level()).c_str());
		printf("  registrant: %s\n", cmd->get_registrant().c_str());
		printf("  admin contact: %s\n", cmd->get_admin_contact().c_str());
		
		if (!cmd->get_trademark_id().empty()) {
			printf("  trademark id: %s\n", cmd->get_trademark_id().c_str());
		}

		if (!cmd->get_trademark_country().empty()) {
			printf("  trademark country: %s\n", cmd->get_trademark_country().c_str());
		}

		if (!cmd->get_trademark_date().empty()) {
			printf("  trademark date: %s\n", cmd->get_trademark_date().c_str());
		}

		if (cmd->get_period().time != 0) {
			printf("  period: %d %s\n", cmd->get_period().time,
			       cmd->get_period().unit.c_str());
		}

		SheppPrint::authInfo(cmd->get_authInfo());
	} // _debug

	if (process_action(act) != 0) {
		return -1;
	}

	return 0;
}

/// defensive registration delete command function
/**
   @param args   vector with command-line arguments
   @return 0 if ok, -1 otherwise
*/
int cmd_def_reg_delete(vector<string> &args)
{
	DefRegDelete act;
	DefRegDeleteCmd *cmd = act.get_command();
	string cmd_name = "delete";

	if (args.empty()) {
		return cmd_def_reg_help("no roid specified", cmd_name);
	}

	cmd->set_roid(args[0]);
	args.erase(args.begin());

	if (!args.empty()) {
		return cmd_def_reg_help("invalid syntax near \"" + args[0] + "\"", cmd_name);
	}

	if (_debug) {
		printf("defensive registration to delete: [%s]\n", cmd->get_roid().c_str());
	} // _debug

	if (process_action(act) != 0) {
		return -1;
	}

	return 0;
}

/// defensive registration renew command function
/**
   @param args   vector with command-line arguments
   @return 0 if ok, -1 otherwise
*/
int cmd_def_reg_renew(vector<string> &args)
{
	DefRegRenew act;
	DefRegRenewCmd *cmd = act.get_command();
	string cmd_name = "renew";

	if (args.empty()) {
		return cmd_def_reg_help("no roid specified", cmd_name);
	}

	cmd->set_roid(args[0]);
	args.erase(args.begin());

	while (!args.empty()) {
		if (args[0] == "-expdate") {
			//-expdate
			args.erase(args.begin());
			if (args.empty()) {
				return cmd_domain_help("curExpDate missing", cmd_name);
			}
			cmd->set_cur_exp_date(args[0]);
			args.erase(args.begin());      
		} else if (args[0] == "-period") {
			args.erase(args.begin());      
			if (args.empty()) {
				return cmd_domain_help("period missing", cmd_name);
			}
			string str_time;
			string unit;
			if (SheppStrUtil::split(args[0], str_time, unit, ":", false) != 0) {
				return cmd_domain_help("error setting period", cmd_name);
			}
			cmd->set_period(atoi(str_time.c_str()), unit);
			args.erase(args.begin());
		} else {
			return cmd_domain_help("invalid syntax near \"" + args[0] + "\"",
			                       cmd_name);
		}
	}

	if (_debug) {
		printf("defensive registration to renew: [%s]\n", cmd->get_roid().c_str());
		printf("  curExpDate: [%s]\n", cmd->get_cur_exp_date().c_str());
		if (cmd->get_period().time != 0 || cmd->get_period().unit != "") {
			printf("  period: [%d %s]\n",
			       cmd->get_period().time,
			       cmd->get_period().unit.c_str());
		}
	} // _debug

	if (process_action(act) != 0) {
		return -1;
	}

	return 0;
}

/// defensive registration update command function
/**
   @param args   vector with command-line arguments
   @return 0 if ok, -1 otherwise
*/
int cmd_def_reg_update(vector<string> &args)
{
	DefRegUpdate act;
	DefRegUpdateCmd *cmd = act.get_command();
	string cmd_name = "update";

	if (args.empty()) {
		return cmd_def_reg_help("no roid specified", cmd_name);
	}

	cmd->set_roid(args[0]);
	args.erase(args.begin());

	while (!args.empty()) {
		if (args[0] == "-add-status") {
			//status to add
			args.erase(args.begin());
			if (args.empty()) {
				return cmd_domain_help("add-status parameter missing", cmd_name);
			}

			DefRegUpdateCmd::Status st;
			if (SheppObjSet::status(st, args) != 0) {
				return cmd_domain_help("invalid status to add", cmd_name);
			}
			cmd->insert_status_add(st);

		} else if (args[0] == "-rem-status") {
			//status to remove
			args.erase(args.begin());
			if (args.empty()) {
				return cmd_domain_help("rem-status parameter missing", cmd_name);
			}

			DefRegUpdateCmd::Status st(args[0]);
			args.erase(args.begin());

			cmd->insert_status_rem(st);

		} else if (args[0] == "-r") {
			// registrant
			args.erase(args.begin());
			if (args.empty()) {
				return cmd_domain_help("registrant parameter missing", cmd_name);
			}

			cmd->set_registrant(args[0]);
			args.erase(args.begin());

		} else if (args[0] == "-a") {
			// admin contact
			args.erase(args.begin());
			if (args.empty()) {
				return cmd_domain_help("admin contact parameter missing", cmd_name);
			}

			cmd->set_admin_contact(args[0]);
			args.erase(args.begin());

		} else if (args[0] == "-tm") {
			//trademark
			args.erase(args.begin());
			if (args.empty()) {
				return cmd_domain_help("trademark parameter missing", cmd_name);
			}

			cmd->set_trademark_id(args[0]);
			args.erase(args.begin());

		} else if (args[0] == "-tmc") {
			//trademark country
			args.erase(args.begin());
			if (args.empty()) {
				return cmd_domain_help("trademark country parameter missing", cmd_name);
			}

			cmd->set_trademark_country(args[0]);
			args.erase(args.begin());

		} else if (args[0] == "-tmd") {
			//trademark date
			args.erase(args.begin());
			if (args.empty()) {
				return cmd_domain_help("trademark date parameter missing", cmd_name);
			}

			cmd->set_trademark_date(args[0]);
			args.erase(args.begin());

		} else if (args[0] == "-auth") {
			// set authInfo
			AuthInfo auth;
			if (SheppObjSet::authInfo(auth, args) != 0) {
				return cmd_def_reg_help("invalid auth", cmd_name);
			}
			cmd->set_authInfo(auth);
		} else {
			return cmd_def_reg_help("invalid syntax near \"" + args[0] + "\"",
			                        cmd_name);
		}
	}

	if (_debug) {
		printf("updating defensive registration: [%s]\n", cmd->get_roid().c_str());

		set<DefRegUpdateCmd::Status> statuses = cmd->get_status_add();
		set<DefRegUpdateCmd::Status>::const_iterator stit;
		if (!statuses.empty()) {
			printf("  status to add:\n");
		}
		for (stit = statuses.begin(); stit != statuses.end(); stit++) {
			printf("    s   : [%s]\n", (*stit).get_status().c_str());
			printf("    lang: [%s]\n", (*stit).get_lang().c_str());
			printf("    msg : [%s]\n", (*stit).get_msg().c_str());
		}
		statuses = cmd->get_status_rem();
		if (!statuses.empty()) {
			printf("  status to remove:\n");
		}
		for (stit = statuses.begin(); stit != statuses.end(); stit++) {
			printf("    s   : [%s]\n", (*stit).get_status().c_str());
			printf("    lang: [%s]\n", (*stit).get_lang().c_str());
			printf("    msg : [%s]\n", (*stit).get_msg().c_str());
		}

		printf("  registrant: %s\n", cmd->get_registrant().c_str());
		printf("  admin contact: %s\n", cmd->get_admin_contact().c_str());
		
		if (!cmd->get_trademark_id().empty()) {
			printf("  trademark id: %s\n", cmd->get_trademark_id().c_str());
		}

		if (!cmd->get_trademark_country().empty()) {
			printf("  trademark country: %s\n", cmd->get_trademark_country().c_str());
		}

		if (!cmd->get_trademark_date().empty()) {
			printf("  trademark date: %s\n", cmd->get_trademark_date().c_str());
		}

		SheppPrint::authInfo(cmd->get_authInfo());
	} // _debug

	if (process_action(act) != 0) {
		return -1;
	}

	return 0;
}

/// main defensive registration command
/**
   @param arg   command-line input arguments
   @return 0 if ok, -1 otherwise
*/
int cmd_def_reg(vector<string> &args)
{
	// defensive registration command processing
	if (!args.empty() && !(args[0] == "help")) {
		if (args[0] == "check") {
			args.erase(args.begin());
			return cmd_def_reg_check(args);
		} else if (args[0] == "info") {
			args.erase(args.begin());
			return cmd_def_reg_info(args);
		} else if (args[0] == "transfer") {
			args.erase(args.begin());
			return cmd_def_reg_transfer(args);
		} else if (args[0] == "create") {
			args.erase(args.begin());
			return cmd_def_reg_create(args);
		} else if (args[0] == "delete") {
			args.erase(args.begin());
			return cmd_def_reg_delete(args);
		} else if (args[0] == "renew") {
			args.erase(args.begin());
			return cmd_def_reg_renew(args);
		} else if (args[0] == "update") {
			args.erase(args.begin());
			return cmd_def_reg_update(args);
		} else {
			return cmd_def_reg_help("invalid command: defreg " + args[0]);
		}
	}

	return cmd_def_reg_help("");
}

#endif // __DEF_REG_FUNCTIONS_H__
