/* 
 * Copyright (C) 2006-2021 Registro.br. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 * 1. Redistribution of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY REGISTRO.BR ``AS IS AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIE OF FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL REGISTRO.BR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */
/* $Id$ */
/** @file RGPRestore.H
 *  @brief EPP/RGP Restore information class
 */

#ifndef __RGP_RESTORE_H__
#define __RGP_RESTORE_H__

#include <string>

#include "libepp_nicbr.H"

using std::string;

LIBEPP_NICBR_NS_BEGIN

/// EPP RGPReport Class
class RGPReport
{
public:
	/// Default constructor
	RGPReport()
	{
		this->reset();
	}

	/// Sets the pre data
	/**
	   @param pre_data data that existed for the domain name prior to
      the domain name being deleted
	 */
	void set_pre_data(const string &pre_data)
	{
		_pre_data = pre_data;
	}

	/// Returns the pre data
	/**
	   @return data that existed for the domain name prior to the domain
      name being deleted
	 */
	string get_pre_data() const
	{
		return _pre_data;
	}

	/// Sets the post data
	/**
	   @param post_data data that exists for the domain name at the time
      the restore report is submitted
	 */
	void set_post_data(const string &post_data)
	{
		_post_data = post_data;
	}

	/// Returns the post data
	/**
	   @return data that exists for the domain name at the time the
      restore report is submitted
	 */
	string get_post_data() const
	{
		return _post_data;
	}

	/// Sets the del time
	/**
	   @param del_time date and time when the domain name delete request
      was sent to the server
	 */
	void set_del_time(const string &del_time)
	{
		_del_time = del_time;
	}

	/// Returns the del time
	/**
	   @return date and time when the domain name delete request was
      sent to the server
	 */
	string get_del_time() const
	{
		return _del_time;
	}

	/// Sets the res time
	/**
	   @param res_time date and time when the original <rgp:restore>
      command was sent to the server
	 */
	void set_res_time(const string &res_time)
	{
		_res_time = res_time;
	}

	/// Returns the res time
	/**
	   @return date and time when the original <rgp:restore> command was
      sent to the server
	 */
	string get_res_time() const
	{
		return _res_time;
	}

	/// Sets the res reason
	/**
	   @param res_reason brief explanation of the reason for restoring
      the domain name
	 */
	void set_res_reason(const string &res_reason)
	{
		_res_reason = res_reason;
	}

	/// Returns the res reason
	/**
	   @return brief explanation of the reason for restoring the domain
      name
	 */
	string get_res_reason() const
	{
		return _res_reason;
	}

	/// Sets the statement
	/**
	   @param statement1 text statement that the client has not restored
      the domain name in order to assume the rights to use or sell the
      domain name for itself or for any third party
	 */
	void set_statement1(const string &statement1)
	{
		_statement1 = statement1;
	}

	
	/// Sets the statement and language
	/**
	   @param statement1 text statement that the client has not restored
	    the domain name in order to assume the rights to use or sell the
	    domain name for itself or for any third party
     @param lang statement1 language
	 */
	void set_statement1(const string &statement1, const string &lang)
	{
		_statement1 = statement1;
		_statement1_lang = lang;
	}

	/// Returns the statement
	/**
	   @return text statement that the client has not restored the
      domain name in order to assume the rights to use or sell the
      domain name for itself or for any third party
	 */
	string get_statement1() const
	{
		return _statement1;
	}

	/// Returns the statement language
	/**
	   @return statement language
	 */
	string get_statement1_lang() const
	{
		return _statement1_lang;
	}

	/// Sets the statement
	/**
	   @param statement2 text statement that the information in the
      restore report is factual to the best of the client's knowledge
	 */
	void set_statement2(const string &statement2)
	{
		_statement2 = statement2;
	}

	/// Sets the statement and language
	/**
	   @param statement2 text statement that the information in the
      restore report is factual to the best of the client's knowledge
     @param lang statement language
	 */
	void set_statement2(const string &statement2, const string &lang)
	{
		_statement2 = statement2;
		_statement2_lang = lang;
	}

	/// Returns the statement
	/**
	   @return text statement that the information in the restore report
      is factual to the best of the client's knowledge
	 */
	string get_statement2() const
	{
		return _statement2;
	}

	/// Returns the statement language
	/**
	   @return statement language
	 */
	string get_statement2_lang() const
	{
		return _statement2_lang;
	}

	/// Sets the other
	/**
	   @param other any information needed to support the statements
      provided by the client
	 */
	void set_other(const string &other)
	{
		_other = other;
	}

	/// Returns the other
	/**
	   @return any information needed to support the statements provided
      by the client
	 */
	string get_other() const
	{
		return _other;
	}

	/// Reset all object attributes
	void reset()
	{
		_pre_data = "";
		_post_data = "";
		_del_time = "";
		_res_time = "";
		_res_reason = "";
		_statement1 = "";
		_statement1_lang = "";
		_statement2 = "";
		_statement2_lang = "";
		_other = "";
	}

private:
	/// Registration data that existed for the domain name prior to the
	/// domain name being deleted
	string _pre_data;

	/// Registration data that exists for the domain name at the time
	/// the restore report is submitted
	string _post_data;

	/// Date and time when the domain name delete request was sent to
	/// the server
	string _del_time;

	/// Date and time when the original <rgp:restore> command was sent
	/// to the server
	string _res_time;

	/// Brief explanation of the reason for restoring the domain name
	string _res_reason;

	/// Text statement that the client has not restored the domain name
	/// in order to assume the rights to use or sell the domain name for
	/// itself or for any third party
	string _statement1;

	/// Statement1 language (by default en - english)
	string _statement1_lang;

	/// Text statement that the information in the restore report is
	/// factual to the best of the client's knowledge
	string _statement2;

	/// Statement2 language (by default en - english)
	string _statement2_lang;

	/// Information needed to support the statements provided by the
	/// client
	string _other;
};

/// EPP RGPRestore Class
class RGPRestore {
public:
	/// EPP RGPRestore Operation Class
	class Operation
	{
	public:
		/// List of RGP restore operations acording to RFC 3915
		enum Value {
			NONE,
			REQUEST,
			REPORT
		};

		/// Convert the enum value into the text value acording to RFC
		/// 3915
		/**
		   @param value enum value
		   @return text representation acording to RFC 3915
		 */
		static string toString(const Value value)
		{
			switch(value) {
			case NONE:
				break;
			case REQUEST:
				return "request";
			case REPORT:
				return "report";
			}

			return "";
		}

		static Value fromString(const string &value)
		{
			if (value == "request") {
				return REQUEST;
			} else if (value == "report") {
				return REPORT;
			}

			return NONE;
		}
	};

	/// Default constructor
	RGPRestore()
	{
		this->reset();
	}

	/// Sets the operation
	/**
	   @param operation describes the redemption grace period operation
	   being requested
	 */
	void set_operation(const Operation::Value operation)
	{
		_operation = operation;
	}

	/// Returns the operation
	/**
	   @return describes the redemption grace period operation being
	   requested
	 */
	Operation::Value get_operation() const
	{
		return _operation;
	}

	/// Sets the report
	/**
	   @param report extra information in case of a report operation
	 */
	void set_report(const RGPReport &report)
	{
		_report = report;
	}

	/// Returns the report
	/**
	   @return extra information in case of a report operation
	 */
	RGPReport get_report() const
	{
		return _report;
	}

	/// Reset all object attributes
	void reset()
	{
		_operation = Operation::NONE;
		_report.reset();
	}

private:
	/// Describes the redemption grace period operation being requested
	Operation::Value _operation;

	/// Extra information in case of a report operation
	RGPReport _report;
};

LIBEPP_NICBR_NS_END

#endif // __RGP_RESTORE_H__
