/* 
 * Copyright (C) 2006-2021 Registro.br. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 * 1. Redistribution of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY REGISTRO.BR ``AS IS AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIE OF FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL REGISTRO.BR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */
/* $Id$ */
/** @file  Launch.H
 *  @brief EPP LaunchInfoRsp Class
 */

#ifndef __LAUNCH_INFO_RSP_H__
#define __LAUNCH_INFO_RSP_H__

#include <list>
#include <string>

#include "Launch.H"

using std::list;
using std::string;

LIBEPP_NICBR_NS_BEGIN

/// EPP LaunchInfoRsp Class
class LaunchInfoRsp
{
public:
	/// EPP LaunchInfoRsp::Status Class
	class Status
	{
	public:
		/// Possible launch info response status
		enum Value {
			NONE,
			PENDING_VALIDATION,
			VALIDATED,
			INVALID,
			PENDING_ALLOCATION,
			ALLOCATED,
			REJECTED,
			CUSTOM
		};

		/// Convert status to text format
		/**
		 * @param value status value
		 * @return text representation of the status
		 */
		static string toStr(const Value value)
		{
			switch (value) {
			case NONE:
				break;
			case PENDING_VALIDATION:
				return "pendingValidation";
			case VALIDATED:
				return "validated";
			case INVALID:
				return "invalid";
			case PENDING_ALLOCATION:
				return "pendingAllocation";
			case ALLOCATED:
				return "allocated";
			case REJECTED:
				return "rejected";
			case CUSTOM:
				return "custom";
			}

			return "";
		}

		/// Convert text status to value
		/**
		 * @param value text status
		 * @return value representation of the status
		 */
		static Value fromStr(const string &value)
		{
			if (value == "pendingValidation") {
				return PENDING_VALIDATION;

			} else if (value == "validated") {
				return VALIDATED;

			} else if (value == "invalid") {
				return INVALID;
				
			} else if (value == "pendingAllocation") {
				return PENDING_ALLOCATION;

			} else if (value == "allocated") {
				return ALLOCATED;

			} else if (value == "rejected") {
				return REJECTED;

			} else if (value == "custom") {
				return CUSTOM;
			}

			return NONE;
		}
	};

	/// Default constructor
	LaunchInfoRsp()
	{
		reset();
	}

	/// Sets the phase of the launch
	/**
	 * @param phase phase of the launch
	 */
	void set_phase(const LaunchPhase &phase) { _phase = phase; }

	/// Returns the phase of the launch
	/**
	 * @return phase of the launch
	 */
	LaunchPhase get_phase() const { return _phase; }

	/// Sets the application Identifier of the Launch Application
	/**
	 * @param applicationId application Identifier of the Launch
	 * Application
	 */
	void set_applicationId(const string &applicationId) { _applicationId = applicationId; }

	/// Returns the application Identifier of the Launch Application
	/**
	 * @return application Identifier of the Launch Application
	 */
	string get_applicationId() const { return _applicationId; }

	/// Sets the status of the Launch Application
	/**
	 * @param status status of the Launch Application
	 */
	void set_status(const Status::Value status) { _status = status; }

	/// Returns the status of the Launch Application
	/**
	 * @return status of the Launch Application
	 */
	Status::Value get_status() const { return _status; }

	/// Sets the list of marks
	/**
	 * @param marks list of marks
	 */
	void set_marks(const list<SMDMark> &marks) { _marks = marks; }

	/// Add a mark
	/**
	 * @param mark mark
	 */
	void add_mark(const SMDMark &mark) { _marks.push_back(mark); }

	/// Returns the list of marks
	/**
	 * @return list of marks
	 */
	list<SMDMark> get_marks() const { return _marks; }

	/// Reset object attributes
	void reset()
	{
		_phase.reset();
		_applicationId.clear();
		_status = Status::NONE;
		_marks.clear();
	}

private:
	/// launch phase that SHOULD be "claims"
	LaunchPhase _phase;

	/// Application Identifier of the Launch Application
	string _applicationId;

	/// Status of the Launch Application
	Status::Value _status;

	/// List of marks
	list<SMDMark> _marks;
};

LIBEPP_NICBR_NS_END

#endif // __LAUNCH_INFO_RSP_H__
