/* 
 * Copyright (C) 2006-2021 Registro.br. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 * 1. Redistribution of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY REGISTRO.BR ``AS IS AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIE OF FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL REGISTRO.BR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */
/* $Id$ */
/** @file  Launch.H
 *  @brief EPP Launch Phase
 */

#ifndef __LAUNCH_CREATE_CMD_H__
#define __LAUNCH_CREATE_CMD_H__

#include <string>

#include "Launch.H"
#include "ClaimsNotice.H"

using std::string;

LIBEPP_NICBR_NS_BEGIN

/// EPP LaunchCreateCmd Class
class LaunchCreateCmd
{
public:
	/// EPP LaunchCreateCmd::Type Class
	class Type
	{
	public:
		/// Possible type values
		enum Value {
			NONE,
			APPLICATION,
			REGISTRATION
		};

		/// Convert type to text format
		/**
		 * @param value type value
		 * @return text representation of the type
		 */
		static string toStr(const Value value)
		{
			switch (value) {
			case NONE:
				break;
			case APPLICATION:
				return "application";
			case REGISTRATION:
				return "registration";
			}

			return "";
		}

		/// Convert text type to value
		/**
		 * @param value text type
		 * @return value representation of the type
		 */
		static Value fromStr(const string &value)
		{
			if (value == "application") {
				return APPLICATION;

			} else if (value == "registration") {
				return REGISTRATION;
			}

			return NONE;
		}
	};

	// Default constructor
	LaunchCreateCmd()
	{
		reset();
	}

	/// Sets the expected type of object
	/**
	 * @param type expected type of object
	 */
	void set_type(const Type::Value type) { _type = type; }

	/// Returns the expected type of object
	/**
	 * @return expected type of object
	 */
	Type::Value get_type() const { return _type; }

	/// Sets the phase of the launch
	/**
	 * @param phase phase of the launch
	 */
	void set_phase(const LaunchPhase &phase) { _phase = phase; }

	/// Returns the phase of the launch
	/**
	 * @return phase of the launch
	 */
	LaunchPhase get_phase() const { return _phase; }

	/// Sets the notice for Claims period
	/**
	 * @param notice notice for Claims period
	 */
	void set_notice(const ClaimsNotice &notice) { _notice = notice; }

	/// Returns the notice for Claims period
	/**
	 * @return notice for Claims period
	 */
	ClaimsNotice get_notice() const { return _notice; }

	/// Sets the code mark list
	/**
	 * @param codeMarks code mark list
	 */
	void set_codeMarks(const list<CodeMark> &codeMarks) { _codeMarks = codeMarks; }
	
	/// Returns the code mark list
	/**
	 * @return code mark list
	 */
	list<CodeMark> get_codeMarks() const { return _codeMarks; }

	/// Sets the signed mark list
	/**
	 * @param signedMarks signed mark list
	 */
	void set_signedMarks(const list<SMD> &signedMarks)
	{
		_signedMarks = signedMarks;
	}

	/// Returns the signed mark list
	/**
	 * @return signed mark list
	 */
	list<SMD> get_signedMarks() const { return _signedMarks; }

	/// Sets the encoded signed mark list
	/**
	 * @param encodedSignedMarks encoded signed mark list
	 */
	void set_encodedSignedMarks(const list<EncodedSignedMark> &encodedSignedMarks)
	{
		_encodedSignedMarks =encodedSignedMarks;
	}

	/// Returns the encoded signed mark list
	/**
	 * @return encoded signed mark list
	 */
	list<EncodedSignedMark> get_encodedSignedMarks() { return _encodedSignedMarks; }

	/// Reset object attributes
	void reset()
	{
		_type = Type::NONE;
		_phase.reset();
		_notice.reset();
		_codeMarks.clear();
		_signedMarks.clear();
		_encodedSignedMarks.clear();
	}

private:
	/// Expected type of object
	Type::Value _type;

	/// Phase of the launch
	LaunchPhase _phase;

	/// Notice for Claims period
	ClaimsNotice _notice;

	/// Code mark list
	list<CodeMark> _codeMarks;

	/// Signed mark list
	list<SMD> _signedMarks;

	/// Encoded signed mark list
	list<EncodedSignedMark> _encodedSignedMarks;
};

LIBEPP_NICBR_NS_END

#endif // __LAUNCH_CREATE_CMD_H__
