/* 
 * Copyright (C) 2006-2021 Registro.br. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 * 1. Redistribution of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY REGISTRO.BR ``AS IS AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIE OF FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL REGISTRO.BR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */
/* $Id$ */
/** @file  DomainTransferCmd.H
 *  @brief EPP DomainTransferCmd Class
 */

#ifndef __DOMAIN_TRANSFER_CMD_H__
#define __DOMAIN_TRANSFER_CMD_H__

#include <string>

#include "libepp_nicbr.H"

#include "Command.H"
#include "CommonData.H"
#include "RegistrationPeriod.H"

using std::string;

LIBEPP_NICBR_NS_BEGIN

/// EPP DomainTransferCmd Class
class DomainTransferCmd : public Command
{
public:
	/// Default constructor
	DomainTransferCmd(bool reset = true) : Command(false)
	{
		if (reset) {
			this->reset();
		}
	}

	/// Sets transfer operation to be performed
	/**
	 * @param operation transfer operation to be performed
	 */
	void set_operation(const TransferOperation::Value operation) { _operation = operation; }

	/// Returns transfer operation to be performed
	/**
	 * @return transfer operation to be performed
	 */
	TransferOperation::Value get_operation() const { return _operation; }

	/// Sets domain name
	/**
	   @param name   fully qualified domain name
	*/
	void set_name(const string& name) { _name = name; }

	/// Returns domain name
	/**
	   @return fully qualified domain name
	*/
	string get_name() const { return _name; }

	/// Sets the number of units to be added to the registration period
	/// of the domain object at completion of the transfer process
	/**
	 * @param time   amount of time
	 * @param unit   measure unit
	 */
	void set_period(const int time, const string& unit)
	{
		_period.time = time;
		_period.unit = unit;
	}

	/// Returns the number of units to be added to the registration
	/// period of the Defensive Registration object at completion of the
	/// transfer process
	/**
	 * @return number of units to be added to the registration period
	 */
	RegistrationPeriod get_period() const { return _period; }

	/// Sets authorization information
	/**
	   @param authInfo   domain authorization information
	*/
	void set_authInfo(const AuthInfo &authInfo)
	{
		_authInfo = authInfo;
	}

	/// Returns authorization information
	/**
	   @return authorization information
	*/
	AuthInfo get_authInfo() const { return _authInfo; }

	/// Reset object attributes
	void reset()
	{
		Command::reset();
		_operation = TransferOperation::NONE;
		_name = "";
		_period.time = 0;
		_period.unit = "";
		_authInfo.reset();
	}
  
protected:
	/// identifies the transfer operation to be performed
	TransferOperation::Value _operation;

	/// fully qualified domain name
	string _name;

	// number of units to be added to the registration period of the
	// domain object at completion of the transfer process
	RegistrationPeriod _period;

	/// authorization information
	AuthInfo _authInfo;
};

LIBEPP_NICBR_NS_END
#endif // __DOMAIN_TRANSFER_CMD_H__
