/* 
 * Copyright (C) 2006-2021 Registro.br. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 * 1. Redistribution of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY REGISTRO.BR ``AS IS AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIE OF FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL REGISTRO.BR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */
/* $Id$ */
/** @file ContactUpdateCmd.H
 *  @brief EPP ContactUpdateCmd Class
 */

#ifndef __CONTACT_UPDATE_CMD_H__
#define __CONTACT_UPDATE_CMD_H__

#include <string>
#include <set>
#include <memory>

#include "libepp_nicbr.H"

#include "Command.H"
#include "CommonData.H"

using std::string;
using std::set;

LIBEPP_NICBR_NS_BEGIN

/// EPP ContactUpdateCmd Class
class ContactUpdateCmd : public Command
{
public:

	/// Default constructor
	ContactUpdateCmd(bool reset = true) : Command(false)
	{
		if (reset) {
			this->reset();
		}
	}

	/// Inserts a status to the object
	/**
	   @param status_add status to be added to the object
	*/
	void insert_status_list_add(const string &status_add) 
	{ 
		_status_list_add.insert(status_add); 
	}
  
	/// Removes a status from the object
	/**
	   @param status_rem status to be removed to the object
	*/
	void insert_status_list_rem(const string &status_rem) 
	{ 
		_status_list_rem.insert(status_rem); 
	}

	/// Returns the list of status to be added to the object
	/**
	   @return status list to be added to the object
	*/
	set<string> get_status_list_add() const { return _status_list_add; }

	/// Returns the list of status to be removed from the object
	/**
	   @return status list to be removed to the object
	*/
	set<string> get_status_list_rem() const { return _status_list_rem; }

	/// Sets the common data object
	/**
	   @param common common data object
	*/
	void set_common_data(const CommonData &common) 
	{
		_common = common;
	}

	/// Returns the common data object
	/**
	   @return a common data object raw pointer
	*/
	CommonData get_common_data() const { return _common; }

	/// Sets authorization information
	/**
	   @param authInfo   domain authorization information
	*/
	void set_authInfo(const AuthInfo &authInfo)
	{
		_authInfo = authInfo;
	}

	/// Returns authorization information
	/**
	   @return authorization information
	*/
	AuthInfo get_authInfo() const { return _authInfo; }

	/// Sets password attribute
	/**
	   @param password   Contact password
	*/
	void set_password(const string &password)
	{
		_password = password;
	}

	/// Returns the password
	/**
	   @return password attribute
	*/
	string get_password() const
	{
		return _password;
	}

	/// Sets reminder attribute
	/**
	   @param reminder   Contact reminder
	*/
	void set_reminder(const string &reminder)
	{
		_reminder = reminder;
	}

	/// Returns the reminder
	/**
	   @return reminder attribute
	*/
	string get_reminder() const
	{
		return _reminder;
	}

	/// Sets language attribute
	/**
	   @param language   Contact language
	*/
	void set_language(const string &language)
	{
		_language = language;
	}

	/// Returns the language
	/**
	   @return language attribute
	*/
	string get_language() const
	{
		return _language;
	}

	/// Sets the properties to be associated with the contact
	/**
	   @param vector of properties
	*/
	void set_properties_add(const vector<string>& properties)
	{
		_properties_add = properties;
	}

	/// Adds a property to the vector of properties
	/**
	   @param a property
	*/
	void insert_property_add(const string& property)
	{
		_properties_add.push_back(property);
	}

	/// Returns the vector of properties that are going to be associated
	/// with the contact
	/**
	   @return the properties
	*/
	vector<string> get_properties_add()
	{
		return _properties_add;
	}

	/// Sets the properties that are going to be removed of the contact
	/**
	   @param vector of properties
	*/
	void set_properties_rem(const vector<string>& properties)
	{
		_properties_rem = properties;
	}

	/// Adds a property to be removed from the contact
	/**
	   @param a property
	*/
	void insert_property_rem(const string& property)
	{
		_properties_rem.push_back(property);
	}

	/// Returns the vector of properties that are going to be removed of
	/// the contact
	/**
	   @return the properties
	*/
	vector<string> get_properties_rem()
	{
		return _properties_rem;
	}
  
	/// Check if there is any extension
	bool has_extension() const {
		return has_lacnic_contact_extension();
	}

	/// Check if there is lacnic contact extension
	bool has_lacnic_contact_extension() const {
		return !_password.empty() || 
			!_reminder.empty() || 
			!_language.empty() || 
			!_properties_add.empty() ||
			!_properties_rem.empty();
	}

	/// Resets the data in the object
	void reset() 
	{ 
		Command::reset();
		_status_list_add.clear();
		_status_list_rem.clear();
		_common.reset();
		_authInfo.reset();
		_password = "";
		_reminder = "";
		_language = "";
		_properties_add.clear();
		_properties_rem.clear();
	}

protected:
	/// Status List to be added to the object
	set<string> _status_list_add;

	/// Status list to be removed to the object
	set<string> _status_list_rem;

	/// Common data object
	CommonData _common;

	/// authorization information
	AuthInfo _authInfo;

	/// contact's password (lacnic contact extension)
	string _password;

	/// contact's tip to remember the password (lacnic contact extension)
	string _reminder;

	/// contact's preferred language (lacnic contact extension)
	string _language;

	/// properties to be added to the object (lacnic contact extension)
	vector<string> _properties_add;

	/// properties to be removed of the object (lacnic contact extension)
	vector<string> _properties_rem;
};

LIBEPP_NICBR_NS_END
#endif //__CONTACT_UPDATE_CMD_H__
