/* 
 * Copyright (C) 2006-2021 Registro.br. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 * 1. Redistribution of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY REGISTRO.BR ``AS IS AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIE OF FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL REGISTRO.BR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */
/* $Id$ */
/** @file BrOrgCreateCmd.H
 *  @brief EPP BrOrgCreateCmd Class
 */
#ifndef __BR_ORG_CREATE_CMD_H__
#define __BR_ORG_CREATE_CMD_H__

#include <string>
#include <map>
#include <vector>

#include "libepp_nicbr.H"

#include "ContactCreateCmd.H"

using std::string;
using std::map;
using std::less;
using std::vector;

LIBEPP_NICBR_NS_BEGIN

/// EPP BrOrgCreateCmd Class
class BrOrgCreateCmd : public ContactCreateCmd
{
public:
	/// Default constructor
	BrOrgCreateCmd(bool reset = true) : ContactCreateCmd(false)
	{
		if (reset) {
			this->reset();
		}
	}

	/// Sets Organization attribute
	/**
	   @param organization   Organization Identifier
	*/
	void set_organization(const string &organization)
	{
		_organization = organization;
	}

	/// Returns the Organization
	/**
	   @return organization attribute
	*/
	string get_organization() const
	{
		return _organization;
	}

	/// Sets the responsible for the organization
	/**
	   @param responsible   Person responsible
	*/
	void set_responsible(const string &responsible)
	{
		_responsible = responsible;
		_responsible_f = true;
	}

	/// Returns the responsible for the organization
	/**
	   @return Person responsible
	*/
	string get_responsible() const
	{
		return _responsible;
	}

	/// Sets the responsible change flag
	/**
	   @param responsible_f responsible change flag
	*/
	void set_responsible_f(const bool responsible_f) 
	{
		_responsible_f = responsible_f;
	}
  
	/// Returns the responsible change flag
	/**
	   @return The responsible change flag
	*/
	bool get_responsible_f() const
	{
		return _responsible_f;
	}

	/// Inserts a Contact into the list
	/**
	   @param id Contact Id
	   @param type Contact Type
	*/
	void insert_contact(const string &type, const string &id)
	{
		_contact_list[type] = id;
	}

	/// Returns the Contact List
	/**
	   @return Contact List
	*/
	map< string, string, less<string> > get_contact_list() const
	{
		return _contact_list;
	}

	/////////////////////////////////////////////
	// LACNIC Org extension methods

	/// Sets the organization type
	/**
	   @param type organization type
	 */
	void set_type(const string &type)
	{
		_type = type;
	}

	/// Returns the organization type
	/**
	   @return organization type
	 */
	string get_type() const
	{
		return _type;
	}

	/// Sets the EPP password
	/**
	   @param password EPP password
	 */
	void set_epp_password(const string &password)
	{
		_epp_password = password;
	}

	/// Returns the EPP password
	/**
	   @return EPP password
	 */
	string get_epp_password() const
	{
		return _epp_password;
	}

	/// Inserts an EPP IP or range that will be allowed in the EPP
	/// server
	/*
	  @param ip IP or range
	 */
	void insert_epp_ip(const string &ip)
	{
		_epp_ips.push_back(ip);
	}

	/// Returns list of IPs or ranges that will be allowed in the EPP
	/// server
	/*
	  @return list of EPP IPs or ranges
	 */
	vector<string> get_epp_ips() const
	{
		return _epp_ips;
	}

	/// Inserts a renewal type
	/*
	  @param type renewal type
	 */
	void insert_renewal_type(const string &type)
	{
		_renewal_types.push_back(type);
	}

	/// Sets the organization category and characteristics
	/*
	  @param types renewal types
	 */
	void set_renewal_types(const vector<string> &types)
	{
		_renewal_types = types;
	}

	/// Returns list of renewal types of an organization
	/*
	  @return list of renewal types of an organization
	 */
	vector<string> get_renewal_types() const
	{
		return _renewal_types;
	}

	/// Sets the organization resources class
	/*
	  @param resources_class for now can be "all-resources" or "non-legacy-only"
	 */
	void set_resources_class(const string &resources_class)
	{
		_resources_class = resources_class;
	}

	/// Returns the organization resources class
	/*
	  @return organization resources class
	 */
	string get_resources_class() const
	{
		return _resources_class;
	}

	/// reset attributes
	void reset()
	{
		ContactCreateCmd::reset();
		_organization = "";
		_responsible = "";
		_responsible_f = false;
		_contact_list.clear();
		_type = "";
		_epp_password = "";
		_epp_ips.clear();
		_renewal_types.clear();
		_resources_class = "";
	}
  
protected:
	/// map with organization attribute
	string _organization;

	/// Person responsible for the organization
	string _responsible;

	/// Responsible change flag
	bool _responsible_f;

	/// Human Contacts
	map<string, string, less<string> > _contact_list;

	/////////////////////////////////////////////
	// LACNIC Org extension methods

	/// Define the organization type
	string _type;

	/// EPP password for authentication
	string _epp_password;

	/// IP addresses or ranges that are allowed to connect to the EPP
	/// interface
	vector<string> _epp_ips;

	/// List of category and characteristics of the organization
	vector<string> _renewal_types;

	/// There are some options when categorizing an organization. We can
	/// use all resouces to categorize or only the non legacy
	/// resources. The possible values are "all-resources" and
	/// "non-legacy-only"
	string _resources_class;
};

LIBEPP_NICBR_NS_END
#endif //__BR_ORG_CREATE_CMD_H__
